function [A, B, r] = svd_cca(X, Y, K)
    % Input: 
    % X - matrix of dimensions NxM1
    % Y - matrix of dimensions NxM2
    % K - number of canonical components

    % Output:
    % A - matrix of dimensions M1xK, where K is the number of canonical components
    % B - matrix of dimensions M2xK, where K is the number of canonical components
    % r - vector of length K, containing the canonical correlations

    % Normalization
    X = X - mean(X, 1);
    Y = Y - mean(Y, 1);

    % Calculate the covariance matrices
    Cxx = cov(X);
    Cyy = cov(Y);
    Cxy = X' * Y / (size(X, 1) - 1);
    Cyx = Y' * X / (size(X, 1) - 1);

    % Perform singular value decomposition (SVD)
    [U, ~, ~] = svd(sqrtm(inv(Cxx))' * Cxy * sqrtm(inv(Cyy)));
    [V, ~, ~] = svd(sqrtm(inv(Cyy))' * Cyx * sqrtm(inv(Cxx)));
    U = U(:, 1:K);
    V = V(:, 1:K);

    % Calculate the canonical coefficients
    A = sqrtm(inv(Cxx)) * U;
    B = sqrtm(inv(Cyy)) * V;

    % Normalize the canonical vectors
    A = A * diag(1./sqrt(sum(A.^2)));
    B = B * diag(1./sqrt(sum(B.^2)));

    % Calculate the canonical correlations
    r = diag(corr(X*A,Y*B));
    for i = 1:K
        A(:, i) = A(:, i) * sign(r(i));
    end
    r = abs(r);
end